// Copyright 2014 Google Inc. All Rights Reserved.

#include "common.h"
#include "BluetoothEndpoint.h"

void BluetoothEndpoint::registerCarBluetoothAddress(
        const string& carBluetoothAddress) {
    mCarBluetoothAddress.assign(carBluetoothAddress);
}

void BluetoothEndpoint::registerPairingMethod(
        const BluetoothPairingMethod& pairingMethod) {
    mSupportedPairingMethods.insert(pairingMethod);
}

void BluetoothEndpoint::registerCallbacks(
        const shared_ptr<IBluetoothCallbacks>& callbacks) {
    mCallbacks = callbacks;
}

void BluetoothEndpoint::requestDelayedPairing() {
    sendPairingResponse(STATUS_BLUETOOTH_PAIRING_DELAYED, false);
}

void BluetoothEndpoint::onReadyForPairing(bool alreadyPaired) {
    sendPairingResponse(STATUS_SUCCESS, alreadyPaired);
}

void BluetoothEndpoint::sendAuthData(const string& authData) {
    BluetoothAuthenticationData authDataMessage;
    authDataMessage.set_auth_data(authData);
    IoBuffer buf;
    mRouter->marshallProto(BLUETOOTH_MESSAGE_AUTHENTICATION_DATA,
                           authDataMessage, &buf);
    queueOutgoing(buf.raw(), buf.size());
}

void BluetoothEndpoint::addDiscoveryInfo(ServiceDiscoveryResponse* sdr) {
    Service* service = sdr->add_services();
    service->set_id(id());
    BluetoothService* btService = service->mutable_bluetooth_service();

    btService->set_car_address(mCarBluetoothAddress);

    for(set<BluetoothPairingMethod>::iterator it =
                    mSupportedPairingMethods.begin();
            it != mSupportedPairingMethods.end();
            ++it) {
        btService->add_supported_pairing_methods(*it);
    }
}

int BluetoothEndpoint::routeMessage(
        uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg) {
    int ret = STATUS_UNEXPECTED_MESSAGE;
    switch (type) {
        case BLUETOOTH_MESSAGE_PAIRING_REQUEST:
            // The first sizeof(uint16_t) bytes is the 'type'. So skip it.
            // TODO(edjee): This code is common in all the endpoints'
            // routeMessage functions. Refactor it.
            uint8_t* ptr = (uint8_t*) msg->raw() + sizeof(uint16_t);
            size_t len = msg->size() - sizeof(uint16_t);
            BluetoothPairingRequest req;
            if (PARSE_PROTO(req, ptr, len)) {
                handlePairingRequest(req);
                return STATUS_SUCCESS;
            }
    }
    return ret;
}

void BluetoothEndpoint::handlePairingRequest(
        const BluetoothPairingRequest& req) {
    const string& phoneAddress = req.phone_address();
    const BluetoothPairingMethod pairingMethod = req.pairing_method();
    mCallbacks->onPairingRequest(phoneAddress, pairingMethod);
}

void BluetoothEndpoint::sendPairingResponse(int32_t status,
                                            bool alreadyPaired) {
    BluetoothPairingResponse resp;
    resp.set_status(static_cast<MessageStatus>(status));
    resp.set_already_paired(alreadyPaired);
    IoBuffer buf;
    mRouter->marshallProto(BLUETOOTH_MESSAGE_PAIRING_RESPONSE, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
}
